import { objectMapEntries } from "@tldraw/editor";
import { useMemo } from "react";
import { PORTRAIT_BREAKPOINT } from "./constants.mjs";
import { useBreakpoint } from "./context/breakpoints.mjs";
import { useDialogs } from "./context/dialogs.mjs";
import { useToasts } from "./context/toasts.mjs";
import { useTranslation } from "./hooks/useTranslation/useTranslation.mjs";
function useDefaultHelpers() {
  const { addToast, removeToast, clearToasts } = useToasts();
  const { addDialog, clearDialogs, removeDialog, updateDialog } = useDialogs();
  const breakpoint = useBreakpoint();
  const isMobile = breakpoint < PORTRAIT_BREAKPOINT.TABLET_SM;
  const msg = useTranslation();
  return useMemo(
    () => ({
      addToast,
      removeToast,
      clearToasts,
      addDialog,
      clearDialogs,
      removeDialog,
      updateDialog,
      msg,
      isMobile
    }),
    [
      addDialog,
      addToast,
      clearDialogs,
      clearToasts,
      msg,
      removeDialog,
      removeToast,
      updateDialog,
      isMobile
    ]
  );
}
function mergeOverrides(overrides, defaultHelpers) {
  const mergedTranslations = {};
  for (const override of overrides) {
    if (override.translations) {
      for (const [key, value] of objectMapEntries(override.translations)) {
        let strings = mergedTranslations[key];
        if (!strings) {
          strings = mergedTranslations[key] = {};
        }
        Object.assign(strings, value);
      }
    }
  }
  return {
    actions: (editor, schema) => {
      for (const override of overrides) {
        if (override.actions) {
          schema = override.actions(editor, schema, defaultHelpers);
        }
      }
      return schema;
    },
    tools: (editor, schema, helpers) => {
      for (const override of overrides) {
        if (override.tools) {
          schema = override.tools(editor, schema, { ...defaultHelpers, ...helpers });
        }
      }
      return schema;
    },
    translations: mergedTranslations
  };
}
function useShallowArrayEquality(array) {
  return useMemo(() => array, array);
}
function useMergedTranslationOverrides(overrides) {
  const overridesArray = useShallowArrayEquality(
    overrides == null ? [] : Array.isArray(overrides) ? overrides : [overrides]
  );
  return useMemo(() => {
    const mergedTranslations = {};
    for (const override of overridesArray) {
      if (override.translations) {
        for (const [key, value] of objectMapEntries(override.translations)) {
          let strings = mergedTranslations[key];
          if (!strings) {
            strings = mergedTranslations[key] = {};
          }
          Object.assign(strings, value);
        }
      }
    }
    return mergedTranslations;
  }, [overridesArray]);
}
function useMergedOverrides(overrides) {
  const defaultHelpers = useDefaultHelpers();
  const overridesArray = useShallowArrayEquality(
    overrides == null ? [] : Array.isArray(overrides) ? overrides : [overrides]
  );
  return useMemo(
    () => mergeOverrides(overridesArray, defaultHelpers),
    [overridesArray, defaultHelpers]
  );
}
export {
  mergeOverrides,
  useDefaultHelpers,
  useMergedOverrides,
  useMergedTranslationOverrides
};
//# sourceMappingURL=overrides.mjs.map
